<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');

// Optional config file (DO NOT commit secrets). Copy api/config.sample.php to api/config.php and edit.
$config = [
  'storage' => 'file', // 'file' or 'mysql'
  'file_path' => __DIR__ . '/../data/claims.jsonl',
  'mysql' => [
    'dsn' => '',
    'user' => '',
    'pass' => '',
  ],
];

$configFile = __DIR__ . '/config.php';
if (is_file($configFile)) {
  $loaded = require $configFile;
  if (is_array($loaded)) {
    $config = array_replace_recursive($config, $loaded);
  }
}

function respond(int $status, array $payload): void {
  http_response_code($status);
  echo json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

function readJsonBody(): array {
  $raw = file_get_contents('php://input');
  if ($raw === false || trim($raw) === '') return [];
  $data = json_decode($raw, true);
  return is_array($data) ? $data : [];
}

function requiredString(array $data, string $key, int $maxLen = 200): string {
  $val = $data[$key] ?? '';
  $val = is_string($val) ? trim($val) : '';
  if ($val === '') return '';
  if (mb_strlen($val) > $maxLen) $val = mb_substr($val, 0, $maxLen);
  return $val;
}

function isValidEmail(string $email): bool {
  return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

try {
  if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
    respond(405, ['ok' => false, 'error' => 'Method not allowed']);
  }

  $data = readJsonBody();

  $ticketId = requiredString($data, 'ticketId', 64);
  $fullName = requiredString($data, 'fullName', 120);
  $documentId = requiredString($data, 'documentId', 60);
  $phone = requiredString($data, 'phone', 40);
  $email = requiredString($data, 'email', 120);
  $reward = requiredString($data, 'reward', 120);
  $createdAt = requiredString($data, 'createdAt', 40);

  if ($ticketId === '' || $fullName === '' || $documentId === '' || $phone === '' || $email === '' || $reward === '') {
    respond(400, ['ok' => false, 'error' => 'Missing required fields']);
  }
  if (!isValidEmail($email)) {
    respond(400, ['ok' => false, 'error' => 'Invalid email']);
  }

  $record = [
    'ticketId' => $ticketId,
    'fullName' => $fullName,
    'documentId' => $documentId,
    'phone' => $phone,
    'email' => $email,
    'reward' => $reward,
    'createdAt' => $createdAt !== '' ? $createdAt : gmdate('c'),
    'ip' => $_SERVER['REMOTE_ADDR'] ?? '',
    'userAgent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
  ];

  if (($config['storage'] ?? 'file') === 'mysql') {
    $dsn = (string)($config['mysql']['dsn'] ?? '');
    $user = (string)($config['mysql']['user'] ?? '');
    $pass = (string)($config['mysql']['pass'] ?? '');
    if ($dsn === '' || $user === '') {
      respond(500, ['ok' => false, 'error' => 'MySQL config missing']);
    }

    $pdo = new PDO($dsn, $user, $pass, [
      PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
      PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);

    $pdo->exec("
      CREATE TABLE IF NOT EXISTS claims (
        id INT AUTO_INCREMENT PRIMARY KEY,
        ticket_id VARCHAR(64) NOT NULL,
        full_name VARCHAR(120) NOT NULL,
        document_id VARCHAR(60) NOT NULL,
        phone VARCHAR(40) NOT NULL,
        email VARCHAR(120) NOT NULL,
        reward VARCHAR(120) NOT NULL,
        created_at VARCHAR(40) NOT NULL,
        ip VARCHAR(64) NOT NULL,
        user_agent VARCHAR(255) NOT NULL
      ) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci
    ");

    $stmt = $pdo->prepare("
      INSERT INTO claims
        (ticket_id, full_name, document_id, phone, email, reward, created_at, ip, user_agent)
      VALUES
        (:ticket_id, :full_name, :document_id, :phone, :email, :reward, :created_at, :ip, :user_agent)
    ");
    $stmt->execute([
      ':ticket_id' => $record['ticketId'],
      ':full_name' => $record['fullName'],
      ':document_id' => $record['documentId'],
      ':phone' => $record['phone'],
      ':email' => $record['email'],
      ':reward' => $record['reward'],
      ':created_at' => $record['createdAt'],
      ':ip' => $record['ip'],
      ':user_agent' => mb_substr($record['userAgent'], 0, 255),
    ]);
  } else {
    $path = (string)($config['file_path'] ?? (__DIR__ . '/../data/claims.jsonl'));
    $dir = dirname($path);
    if (!is_dir($dir)) {
      @mkdir($dir, 0755, true);
    }
    $line = json_encode($record, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n";
    if (@file_put_contents($path, $line, FILE_APPEND | LOCK_EX) === false) {
      respond(500, ['ok' => false, 'error' => 'Unable to write storage file']);
    }
  }

  respond(200, ['ok' => true, 'ticketId' => $ticketId]);
} catch (Throwable $e) {
  respond(500, ['ok' => false, 'error' => 'Server error']);
}

